/*
 * @(#)Juggler.java	
 *
 * Copyright (c) 1996,7 Allen Knutson, Matthew Levine, Gregory Warrington
 * This file is part of [MA][GNU]S.
 * 
 * [MA][GNU]S is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * [MA][GNU]S is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with [MA][GNU]S; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

import java.applet.*;
import java.io.*;
import java.awt.*;
import java.net.*;
import java.util.*;

/**
 * @author  Allen Knutson, Matthew Levine, Gregory Warrington
 * @version $Id: Juggler.java,v 1.112 1997/04/13 22:09:21 mslevine Exp $
 */

// the "main" program for the applet
public class Juggler extends Frame implements Runnable
{
  boolean running, changingpattern, painting, updating, userstopped, 
    startup, resizing, blackandwhite;
  Thread thread;
  Vector visible_objects, moving_objects;  // balls, hands, etc
  Image second_buffer;           // to do double buffered animation
  Graphics second_bufferGC;
  Dimension second_buffer_size;
  FontMetrics fontmetrics;
  Taskmaster taskmaster;
  Pattern pattern;
  TextField patternentry;
  JugglerControls controls;
  Font ballfont, handfont, basefont;
  Timer timer;
  Scale scale;
  double lastdraw;
  long lastclicktime;
  MessageWindow message;
  Orbits orbits;
  Hashtable params;
  SpaceTime spacetime;
  StateGraph stategraph;
  JugglerApplet applet;
  Color bgcolor;
  String version = "[MA][GNU]S v1.01";
  Panel statusline;
  Label status;
  Button details;
  Panel lr;
  Marker ul;

public Juggler()   // constructor (we defer initialization to init())
  {
    applet = null;
    params = new Hashtable();
  }

public Juggler(JugglerApplet a)
  {
    applet = a;
  }

public static void main(String argv[])
  {
    Juggler j = new Juggler();
    int i;
    
    // read ~/.jugglerrc
    String homedir = System.getProperty("user.home", "");
    File rc = new File(homedir, ".jugglerrc");
    if (rc.exists() && rc.canRead())
      {
	try 
	  {
	    FileInputStream rcis = new FileInputStream(rc);
	    StreamTokenizer rcst = new StreamTokenizer(rcis);
	    rcst.slashSlashComments(true);
	    rcst.slashStarComments(true);
	    rcst.quoteChar('"');
	    rcst.ordinaryChars('0', '9');
	    rcst.ordinaryChar('[');
	    rcst.ordinaryChar(']');
	    rcst.ordinaryChar('(');
	    rcst.ordinaryChar(')');
	    rcst.ordinaryChar(',');
	    rcst.ordinaryChar('-');
	    rcst.ordinaryChar('+');
	    rcst.ordinaryChar('_');
	    rcst.wordChars('0', '9');
	    rcst.wordChars('[', '[');
	    rcst.wordChars(']', ']');
	    rcst.wordChars('(', '(');
	    rcst.wordChars(')', ')');
	    rcst.wordChars(',', ',');
	    rcst.wordChars('-', '-');
	    rcst.wordChars('+', '+');
	    rcst.wordChars('_', '_');
	    String first=null;
	    boolean wantsecond = false;
	  rcinput: while (true)
	    {
	      try {i=rcst.nextToken();}
	      catch (IOException ioe) { break; }
	      switch (i)
		{
		case StreamTokenizer.TT_WORD:
		case '"':
		  if (first == null)
		    first = rcst.sval;
		  else
		      if (wantsecond)
			{
			  j.params.put(first, rcst.sval);
			  first = null;
			  wantsecond = false;
			}
		      else
			{
			  j.params.put(first, "");
			  first = rcst.sval;
			}
		    break;
		  case StreamTokenizer.TT_EOF:
		    if (first != null)
		      j.params.put(first, "");
		    break rcinput;
		  case '=':
		    wantsecond= (first != null);
		  }
	      }
	  }
	catch (FileNotFoundException e) {}
      }

    // handle command line args
    for(i=0; i<argv.length; i++)
      {
	int k = argv[i].indexOf('=');
	if (k > 0)
	  if (k < argv[i].length())
	    j.params.put(argv[i].substring(0, k), 
		       argv[i].substring(k+1, argv[i].length()));
	  else
	    j.params.put(argv[i].substring(0, k), "");
	else
	  j.params.put(argv[i], "");
      }

    j.init();
    j.controls.hotlist.initAsHotlist();
    if (j.getParameter("pattern") == null) 
      j.pattern.parseString("(0,0)");
    else
      j.pattern.parseString(j.getParameter("pattern"));
    j.start();
  }

public void init()
  {
    super.resize(330, 400);

    setTitle(version);
    basefont = new Font("Helvetica", Font.PLAIN, 12);
    updating = running = changingpattern = painting = false;
    startup = userstopped = true;
    lastdraw = 0;
    lastclicktime = 0;
    message = new MessageWindow();
    thread = null;

    setLayout(new BorderLayout());
    add("North", ul = new Marker());
    lr = new Panel();
    lr.setLayout(new BorderLayout());
    statusline = new Panel();
    statusline.setLayout(new BorderLayout());
    statusline.add("North", status = new Label("Paused -- Click to start"));
    details = new Button("Details...");
    lr.add("South", statusline);
    lr.add("North", patternentry = new TextField(30));
    add("South", lr);

    timer = new Timer();
    scale = new Scale(this);
    visible_objects = new Vector(20,10);
    moving_objects = new Vector(20,10);
    second_buffer_size = new Dimension();

    pattern = new Pattern(this);
    taskmaster = new Taskmaster(this);
    stategraph = new StateGraph();
    spacetime = new SpaceTime(this);
    orbits = new Orbits(this);
    setMenuBar(controls = new JugglerControls(this));

    Body body = new Body(this, getImage ("pics/body.gif"), 
			 0.0, 0.2, -1000000010, 2,  4, Color.lightGray);
    visible_objects.addElement(body);
    Hand r = new Hand(this, Color.lightGray, 0, body);
    Hand l = new Hand(this, Color.lightGray, 1, body);
    visible_objects.addElement(new Eyes(this, getImage("pics/normaleyes.gif"), 
					.9, .4, body));
    visible_objects.addElement(new Finger(this, Color.darkGray, r));
    visible_objects.addElement(new Finger(this, Color.darkGray, l));
    visible_objects.addElement(r); moving_objects.addElement(r);
    visible_objects.addElement(l); moving_objects.addElement(l);

    String initpatstr = getParameter("pattern");
    if (initpatstr != null)
      {
	patternentry.setText(initpatstr);
	handleEvent(new Event(patternentry, Event.ACTION_EVENT, null));
      }

    String colstr = getParameter("bgcolor");
    int color = 0;
    if (colstr != null)
      try {
        color = Integer.valueOf(colstr, 16).intValue();
      } catch (NumberFormatException e) { }
    bgcolor = new Color(color);
    int depth = Toolkit.getDefaultToolkit().getColorModel().getPixelSize();
    blackandwhite = (depth < 4);

    taskmaster.resetScales();  // is this useful?
    controls.reset(false);  

    // these next several lines are terrible things to do drawing other
    // than how it was intended such that the juggler appears at startup
    Rectangle rul = ul.bounds();
    Rectangle rlr = lr.bounds();
    scale.resize(new Rectangle(rul.x, rul.y, rlr.x+rlr.width-rul.x,
			       rlr.y-rul.y));
    scale.setDrawTime(0);
    taskmaster.updateHand(r); r.move(0); r.path.endtime = 0;
    taskmaster.updateHand(l); l.move(0); l.path.endtime = 0;
    painting = true;
    repaint();

    show();
    //    try {Thread.currentThread().sleep(1000);} catch (InterruptedException e){}
    // controls.reset(true);
    // try {Thread.currentThread().sleep(10);} catch (InterruptedException e){}
    controls.reset(true);

    showStatus("Paused -- Click to start", false);
  }
  
public void start()
  {
    if (userstopped || running) return;
    startup = false;
    timer.start();
    running = true;
    thread = new Thread(this);
    thread.start();
    thread.setPriority(thread.getPriority()-2);
    showStatus("", false);
  }
  
public void stop()
  {
    if (!userstopped || !running) return;
    timer.stop();
    running = false;
    thread.stop();
    thread = null;
    showStatus("Paused -- Click to continue", false);
  }

public void quit()
  {
    if (applet != null)
      {
	stop();
	orbits.hide();
	spacetime.hide();
	stategraph.hide();
	message.hide();
	controls.settings.hide();
	controls.helpframe.hide();
	controls.hideTearOffs();
	hide();
	userstopped = true;
      }
    else 
      {
	controls.saveOptions();
	System.exit(0);
      }
  }

public void update(Graphics g)
  {
    paint(g);
  }

public final void paint(Graphics g)
  {
    Dimension d = size();
    if (d.width != second_buffer_size.width ||
        d.height != second_buffer_size.height)
      {
        second_buffer = createImage(d.width, d.height);
        second_bufferGC = second_buffer.getGraphics();
        second_buffer_size = d;
        Rectangle rul = ul.bounds();
        Rectangle rlr = lr.bounds();
        scale.resize(new Rectangle(rul.x, rul.y, rlr.x+rlr.width-rul.x,
                                   rlr.y-rul.y));
	scale.setDrawTime(timer.now());
        g.setColor(bgcolor);
        g.fillRect(0, 0, d.width, d.height);
      }           

    if (!controls.showtrails.getState()) 
      {
	second_bufferGC.setColor(bgcolor);
	second_bufferGC.fillRect(0, 0, d.width, d.height); 
      }
    mypaint(second_bufferGC);
    g.drawImage(second_buffer, 0, 0, this);
  }

public void mypaint(Graphics g)
  {
    sortObjectsByZ();  // smallest z drawn first
    for(int i=visible_objects.size()-1; i>=0; i--)
      {
	Spriggan t = (Spriggan)visible_objects.elementAt(i);
	if (t instanceof Ball || t instanceof Hand) t.paint(g);
	else if (controls.showjuggler.getState()) t.paint(g);
      }
    painting = false;
  }

  // sort by z for drawing (smallest first)
private void sortObjectsByZ()
  {
    int i, j, n;
    boolean changed = true;

    if(!pattern.curSeq.multi)
      return;

    n = visible_objects.size()-1;
    for(i=1; i<=n && changed; i++)
      {
	changed = false;
	for(j=n; j>=i; j--)
	  {
	    Spriggan t1 = (Spriggan)visible_objects.elementAt(j);
	    Spriggan t2 = (Spriggan)visible_objects.elementAt(j-1);
	    if (t1.pos.z > t2.pos.z)
	      {
		visible_objects.setElementAt(t2, j);
		visible_objects.setElementAt(t1, j-1);
		changed = true;
	      }
	  }
      }
  }

  // put balls that will be held last, so that they getthe right info
  // on whether or not to move
  private void sortByHold()
  {
    int i, j, n;
    boolean changed = true;

    if(!pattern.curSeq.holds)
      return;

    n = moving_objects.size()-1;
    for(i=1; i<=n && changed; i++)
      {
	changed = false;
	for(j=n; j>=i; j--)
	  {
	    Spriggan t1 = (Spriggan)moving_objects.elementAt(j);
	    Spriggan t2 = (Spriggan)moving_objects.elementAt(j-1);
	    if(t1.compare(t2) > 0)
	      {
		moving_objects.setElementAt(t2, j);
		moving_objects.setElementAt(t1, j-1);
		changed = true;
	      }
	  }
      }
  }

  // the main event loop
public void run()
  {
    while (running)
      {
	if (changingpattern) 
	  {
	    try {Thread.currentThread().sleep(10); }
	    catch (InterruptedException e) { return;}
	    continue;
	  }
	if(pattern.dirty)
	  {
	    sortByHold();
	    sortObjectsByZ();
	    pattern.curSeq.setMultiplexes();
	    pattern.curSeq.setHolds();
	    pattern.dirty = false;
	    //System.out.println("dirty");
	  }
	updating = true;
	//spamMemory();
	double now = timer.now();
	scale.setDrawTime(now);
	int n = moving_objects.size();
	do
	  {
	    lastdraw = Math.min(lastdraw+1, now); 
	    pattern.incrTime((int)Math.floor(lastdraw));
	    sortByHold();
	    for(int i=0; i<n; i++)
	      {
		Spriggan t = (Spriggan)moving_objects.elementAt(i);
		if (!t.move(lastdraw)) 
		  {
		    moving_objects.removeElementAt(i);
		    visible_objects.removeElement(t);
		    i--; n--;
		  }
	      }
	    spacetime.updateFromSpriggans(moving_objects,
					  (int)Math.floor(lastdraw));
	  } while (lastdraw < now);
	updating = false;

	painting = true;
	repaint();
	while (painting)
	  { 
	    // System.out.println("waiting for paint");
	    try { Thread.currentThread().sleep(5); }
	    catch (InterruptedException e) { return;}
	  }
      }
  }

  public void spamMemory()
  {
    Runtime myRuntime = Runtime.getRuntime();

    long free = myRuntime.freeMemory();
    long tot =  myRuntime.totalMemory();
    for (long j=free; j<tot; j+=100000) System.out.print(" ");
    System.out.println(free +" left of "+ tot);
  }

  // pause if clicked
public boolean  mouseDown(Event ev, int x, int y)
  {
    userstopped = !userstopped;
    if (userstopped)
      stop();
    else
      start();
    return true;
  }

public boolean handleEvent(Event ev)
  {
    if (ev.target == patternentry)
      if (ev.id == Event.ACTION_EVENT)
	{
	  if (!newPattern(patternentry.getText()))
	    {
	      message.setMessage(pattern.parser.errormessages[pattern.parser.errcode]);
	      showStatus("Error: Bad Pattern", true);
	    }
	  else message.hide();
	  return true;   // handled event, don't pass it up further
	}
      else if (ev.id == Event.KEY_ACTION)
	if (ev.key == Event.UP)	
	  {  
	    controls.history.savePartialEntry(patternentry.getText());
	    patternentry.setText(controls.history.prev());
	    return true;
	  }
	else if (ev.key == Event.DOWN)
	  {
	    controls.history.savePartialEntry(patternentry.getText());
	    patternentry.setText(controls.history.next());
	    return true;
	  }
	else if (ev.key == Event.F2)
	  {
	    spamMemory();
	    return true;
	  }
    if (ev.target == details && ev.id == Event.ACTION_EVENT)
      {
	if (!message.isVisible()) message.show();
	return true;
      }
    
    if(ev.id == Event.MOUSE_DOWN)
      mouseDown(ev, ev.x, ev.y);

    return controls.handleEvent(ev); /* pass on menu events to controls */
  }

public void random(boolean rand)
  {
    if (rand == pattern.randomPattern) return;

    changingpattern = true;
    while (updating) 
      {
	//  System.out.println("rand: waiting for update/paint to complete");
	try { Thread.currentThread().sleep(12); }
	catch (InterruptedException e) {}
      }

    if(!pattern.randomPattern && pattern.noBalls())
      {
	pattern.addSeq("4");
	transToNewPattern();
	pattern.nextSequence();
      }
    pattern.toggleRandom();
    if (!rand)
      transToNewPattern();
    else
      taskmaster.resetScales();

    changingpattern = false;
  }

public String getAppletInfo()
  {
    return getText("about.txt");
  }
  
  public String[][] getParameterInfo() 
  {
    return controls.getParameterInfo();
  }    

public boolean newPattern(String s)
  {
    boolean ans = false;

    changingpattern = true;
    while (updating) 
      {
	//	System.out.println("waiting for update/paint to complete");
	try { Thread.currentThread().sleep(12); }
	catch (InterruptedException e) {}
      }

    if(pattern.addSeq(s))  // add to front
      {
	if(pattern.randomPattern)  // turn random off
	  pattern.toggleRandom();
	controls.history.notifyNewSequence(pattern.newSeq);
	ans = transToNewPattern();
      }

    changingpattern = false;
    return ans;
  }

  // return true if ok, false if bad pattern
public boolean transToNewPattern()
  {
    Vector thrws = pattern.transitionTo();
    if (thrws == null)
      {
	return false;  // bad pattern
      }
    
    taskmaster.resetScales();  // update size info for new pattern
 
    int n = thrws.size();
    float rand = (float)Math.random();

    for(int i=0; i<n; i++)  // create new balls 
      {
	Ball b = new Ball(this, Color.red);
	taskmaster.initBall(b, (Thrw)thrws.elementAt(i), lastdraw);
	visible_objects.addElement(b);
	moving_objects.addElement(b);

	rand += (float)1./n;
	if (rand > (float)1.) rand -= (float)1.;
	b.hue = rand;
	if (blackandwhite)
	  b.setColor(Color.getHSBColor((float)0., (float)0., 
				       (float)(.33+2*rand/3)));
	else
	  b.setColor(Color.getHSBColor(rand, (float)1.0, (float)1.0));
      }
    
    if (userstopped) 
      {
	userstopped = !userstopped;
	start();
      }
    if (!running) start();

    return true;
  }

  public Image getImage(String s)
  {
    if (applet != null)
      return applet.getImage(s);
    else
      {
	Toolkit t = Toolkit.getDefaultToolkit();
	return t.getImage(s);
      }
  }

public String getText(String s)
  {
    InputStream i=null;
    if (applet != null)
      i = applet.getText(s);
    else
      {
	File f = new File(s);
	if (f.exists() && f.canRead())
	  {
	    try {
	      i = new FileInputStream(f);
	    } catch (FileNotFoundException e) {return "";}
	  }
      }
    
    if (i == null) return "";
    DataInputStream d = new DataInputStream(i);
    String str, ans = "";
    try {str = d.readLine();} catch (IOException e) {return ans;}
    while (str != null)
      {
	ans = ans+str+"\n";
	try {str = d.readLine();} catch (IOException e) {return ans;}
      }
    return ans;
  }


  public String getParameter(String s)
  {
    if (applet != null)
      return applet.getParameter(s);
    else
      return (String)params.get(s);
  }

  public void showStatus(String s, boolean clickable)
  {
    status.setText(s);
    if (clickable)
      {
	if (statusline.countComponents() != 2)
	  {
	    statusline.removeAll();
	    statusline.add("East", details);
	    statusline.add("Center", status);
	    statusline.layout();
	  }
      }
    else
      if (statusline.countComponents() == 2)
	{
	  statusline.removeAll();
	  statusline.add("North", status);
	  statusline.layout();
	}
  }

  public void setSequenceFromHistory(Sequence seq)
  {
    if (seq instanceof BallSequence)
      {
	if(pattern.randomPattern)  // turn random off
	  pattern.toggleRandom();
	pattern.newSeq = (BallSequence)seq;
	// ((BallSequence)seq).printSequence();
	patternentry.setText(seq.name);
	((BallSequence)seq).validate(orbits);  // just to update orbits
	transToNewPattern();
      }
    else if (seq instanceof HandSequence)
      {
	
      }
  }

}
