/*
 * @(#)PatternParser.java	
 *
 * Copyright (c) 1996,7 Allen Knutson, Matthew Levine, Gregory Warrington. 
 * This file is part of [MA][GNU]S.
 * 
 * [MA][GNU]S is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * [MA][GNU]S is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with [MA][GNU]S; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

import java.applet.*;
import java.io.*;
import java.awt.*;
import java.util.Vector;
import java.util.BitSet;

/**
 * @author  Allen Knutson, Matthew Levine, Gregory Warrington
 * @version $Id: PatternParser.java,v 1.21 1997/03/22 03:14:39 gwar Exp $
 */

////////////////////////////////////////////////////////////////////////
// PatternParser
// TODO:
// 1) upDates to curHand mod by 2 - ie, assume two hands.  need to pass
//    info from taskmaster or pattern?
// 2) it will parse a partially synchronous pattern without comment
// 3) make it more consistent as to where it cares whether you have a
//    comma or not.
// 4) test on outrageous inputs
class PatternParser
{
  static final int OddThrwInSync = 1;
  static final int OddLengthSync = 2;
  static final int UnknownChar = 3;
  static final int UnmatchedBracket = 4;
  static final int XInAsync = 5;

  static final String errormessages[] = {
    "Your pattern is syntactically correct but cannot be juggled.\nYou can find out why by looking at the Orbits window.",
    "You can't have an odd height throw in a synchronous pattern.",
    "You can't have an odd number of throws in a synchronous pattern.",
    "There is an unknown character in your pattern.",
    "You have an unmatched [",
    "You can't use the x modifier in an asynchronous pattern"
  };

  int pos;               // which character in input string
  int curHand;           // which hand current throw corresponds to
  int curTime;           // logical time of throw
  int curThrw;           // numerical value of current throw
  boolean spacedOut=false; // whether pattern has spaces
  BitSet exp = new BitSet(4);
  BitSet imp = new BitSet(4);
  BitSet duh = new BitSet(0);

  int errcode;
  
public PatternParser()
  {
  }

public boolean parseString(String str, BallSequence seq)
  {
    int hgt;

    errcode = 0;
    pos = 0;
    curHand = 0;
    curTime = 0;
    curThrw = 0;

    str = str.trim();
    if (str.indexOf(' ') > -1 && upcaseFree(str)) spacedOut = true;
    else spacedOut = false;
    if ((str.indexOf('(') > -1) || (str.indexOf(')') > -1) || 
	(str.indexOf(',') > -1))
      {
	seq.setSync(true);
	str = str.replace('(', ' ');
	str = str.replace(')', ' ');
	str = str.replace(',', ' ');
      }
    else
      seq.setSync(false); 
    
    while (pos < str.length() && Character.isSpace(str.charAt(pos)))
      pos++;

    // we increment pos within the fn calls as more than one
    // character may need to be read for each "term" in the
    // pattern - ie, for multiplexes.
    while (pos < str.length())
      {
	char ch = str.charAt(pos);
	if (seq.isSync())
	  if (!getSynchronous(str, seq)) return false;
	  else curTime += 2;
	else if(ch == '[')
	  {
	    // MULTIPLEX.  parsing a multiplex has been put in another
	    // method as we could conceivably multiplex from within a
	    // synchronous pattern and this if statement would be
	    // enormous
	    if(!getMultiplex(str, seq)) return false;
	    curHand = (curHand + 1) % 2;
	    curTime++;
	  }
	else
	  {
	    if(getNextTerm(str, seq))
	      seq.addThrow(new Thrw(curThrw, curHand, curTime, exp, imp));
	    else
	      return false;
	    curHand = (curHand + 1) % 2;
	    curTime++;
	  }
      }
    return true;
  }  // PatternParser.parseString

  // -----------------------------------------------------------
  // for error checking, assumes leading '[' has not yet been
  // eaten.  reads until it hits a ']'.  returns new value of i.
  // returns 0 on failure.  getSynchronous is entirely analogous
  // except uses '(' and ')'.
  // ---------------------------------------------------------
private boolean getMultiplex(String str, BallSequence seq)
  {
    char ch=' ';
    int hgt;

//    if(ch != '[')
//      return false;   // getMultiplex shouldn't have been called
    
    pos++;  // clear '['
    while(pos < str.length() && (ch = str.charAt(pos)) != ']')
      {
	if(getNextTerm(str, seq) == true)
	  seq.addThrow(new Thrw(curThrw,curHand,curTime, exp, imp));
	else
	  return false;
      }

    if(ch == ']')  // sanity check
      {
	pos++;
      }
    else
      {
	errcode = UnmatchedBracket;
	return false;
      }
    while (pos < str.length() && Character.isSpace(str.charAt(pos)))
      pos++;
    return true;
  }  // PatternParser.getMultiplex

  // -----------------------------------------------------------
  // see getMultiplex.  note that curTime is not incremented as all 
  // the throws read in getSynchronous are ... you guessed it.
  // -----------------------------------------------------------
private boolean getSynchronous(String str, BallSequence seq)
  {
    char ch;
    int hgt;
    int tmpCount = 0;

    while(pos < str.length() && tmpCount < 2)
      {
	if((ch = str.charAt(pos)) == '[')
	  {
	    if(!getMultiplex(str, seq))
	      return false;
	  }
	else
	  if(getNextTerm(str, seq))
	    if ((curThrw % 2) == 0)
	      seq.addThrow(new Thrw(curThrw,curHand,curTime, exp, imp));
	    else
	      {
		errcode = OddThrwInSync;
		return false;
	      }
	  else
	    return false;
	tmpCount++;
	curHand = (curHand + 1) % 2;
      }
    
    if (tmpCount < 2)
      {
	errcode = OddLengthSync;
	return false;
      }
    else return true;
  } // PatternParser.getSynchronous


  // -------------------------------------------
  // eats trailing space and commas.  for the sequence 11 to be
  // interpreted as B rather than two consecutive 1's, there must be
  // at least one space in the pattern.  Note that if there is any
  // space in the pattern at all, all such sequences will be
  // interpreted in this manner.
  // TODO: doesn't check that decimal throws are valid-looking
  // -------------------------------------------
private boolean getNextTerm(String str, BallSequence seq)
  {
    char ch;
    int th;

    curThrw = 0;
    exp.and(duh);
    imp.and(duh);
    if(spacedOut)  // read as decimal throw until modifier or space
      {
	if((curThrw = isThrow(ch = str.charAt(pos++))) < 0)
	  {
	    errcode = UnknownChar;
	    return false;
	  }
	while(pos < str.length() && (th = isThrow(ch=str.charAt(pos))) > -1)
	  {
	    curThrw = 10*curThrw + th;
	    pos++;
	  }
      }
    else
      if((curThrw = isThrow(ch = str.charAt(pos++))) == -1)
	{
	  errcode = UnknownChar;
	  return false;
	}
    
    // read modifier string
    while(pos < str.length() && isModifier(ch = str.charAt(pos)))
      {
	if (!seq.isSync() && ch == Thrw.cro && curThrw%2 == 0)
	  {
	    errcode = XInAsync;
	    return false;
	  }
	if(ch == Thrw.cro) exp.set(Thrw.CRO);
	if(ch == Thrw.bou) exp.set(Thrw.BOU);
	if(ch == Thrw.rev) exp.set(Thrw.REV);
	if(ch == Thrw.thr) exp.set(Thrw.THR);
	pos++;
      }

    // might be redundant but so what.
    if(!seq.isSync() && (curThrw % 2) != 0)
      imp.set(Thrw.CRO);
    if(exp.get(Thrw.BOU) || (curThrw != 0 && curThrw != 2) || 
       exp.get(Thrw.CRO))
      imp.set(Thrw.THR);

    // eat spaces
    while(pos<str.length() && Character.isSpace(str.charAt(pos)))
      pos++;

//    System.out.println("next: "+str.charAt(pos));
    return true;
  }  // PatternParser.getNextTerm

  // returns numerical value of throw if valid and -1 on failure
  // --------------------------------
private static int isThrow(char ch)
  {
    // regular throw
    if((ch - '0' < 10) && (ch - '0' >= 0))
      return(ch - '0');
    // insanely high throw - pretend we're in base-35.  Note, 
    // you can only use letters up to 'w' as 'x' is for something
    // else.
    if((ch - 'A' < 26) && (ch - 'A' >= 0))
      return(ch - 'A' + 10);
    return -1;
  }  // PatternParser.isThrow

  // ---------------------------------------
private static boolean isModifier(char ch)
  {
    return Character.isLowerCase(ch);
  }  // PatternParser.isModifier


  // if it looks to you that this function should probably be a little
  // more complicated, well, that's where you're right....
  // -----------------------------------------------
public boolean isPatternValid(Thrw thrws[])
  {
    return true;
  }  // Parser.isPatternValid

// this is extremely similar to getNextTerm.  something should probably
// be done to change this.
public static Thrw getThrw(String str)
  {
    char ch;
    int th;
    int pos = 0;
    int cThrw;
    BitSet eMods = new BitSet(4);
    BitSet iMods = new BitSet(4);

    if((cThrw = isThrow(ch = str.charAt(pos++))) == -1)
      return null;
    
    // read modifier string
    while(pos < str.length() && isModifier(ch = str.charAt(pos)))
      {
	if(ch == Thrw.cro) eMods.set(Thrw.CRO);
	if(ch == Thrw.bou) eMods.set(Thrw.BOU);
	if(ch == Thrw.rev) eMods.set(Thrw.REV);
	if(ch == Thrw.thr) eMods.set(Thrw.THR);
	pos++;
      }

    // might be redundant but so what.
    if(cThrw % 2 == 1)
      iMods.set(Thrw.CRO);
    if(eMods.get(Thrw.BOU) || (cThrw != 0 && cThrw != 2) || 
       eMods.get(Thrw.CRO))
      iMods.set(Thrw.THR);

    // hand and time fields don't contain useful info
    return (new Thrw(cThrw,0,0,eMods,iMods));

  }  // PatternParser.getNextTerm

  // str has uppercase letters?
private boolean upcaseFree(String str)
  {
    int c;
    for(int i=str.length()-1; i>=0; i--)
      {
	c = str.charAt(i);
	if (c >= 'A' && c <= 'Z') return false;
      }
    return true;
  }
}





