/*
 * @(#)SpaceTime.java	
 *
 * Copyright (c) 1996,7 Allen Knutson, Matthew Levine, Gregory Warrington
 * This file is part of [MA][GNU]S.
 * 
 * [MA][GNU]S is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * [MA][GNU]S is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with [MA][GNU]S; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

import java.applet.*;
import java.io.*;
import java.awt.*;
import java.util.Vector;
import java.util.Enumeration;

/**
 * @author  Allen Knutson, Matthew Levine, Gregory Warrington
 * @version $Id: SpaceTime.java,v 1.30 1997/03/22 03:14:39 gwar Exp $
 */

public class SpaceTime extends Frame implements Runnable
{
  Juggler juggler;
  Image second_buffer;
  Graphics second_bufferGC;
  Dimension second_buffer_size, real_size;
  boolean running=false, userstopped = false, updating=false, resizing=false;
  Font font;
  int fontsize;
  FontMetrics fm;
  int length = 13;
  double width = .4;
  double textoffset = .1;
  boolean vertical = false;
  double delta;
  int lastupdate=-1;
  double lastdraw;
  Thread thread;
  Marker ur, ll;  

public SpaceTime(Juggler juggler)
  {
    lastdraw = -1;
    this.juggler = juggler;
    second_buffer_size = new Dimension(-1, -1);
    setTitle("Space-Time Diagram");

    setLayout(new BorderLayout());
    add("North", ur = new Marker());
    add("South", ll = new Marker());

    resize(330, 150);
  }
  
public void start()
  {
    if (running || userstopped) return;
    running=true;
    if (thread == null) thread = new Thread(this);
    thread.start();
    thread.setPriority(thread.getPriority()-2);
  }

public void stop()
  {
    if (!running) return;
    running = false;
    thread.stop();
    thread = null;
  }

public void run()
  {
    while (running)
      {
	try { Thread.sleep(100); } catch (InterruptedException e) {return; }
	if (juggler.timer.now() != lastdraw)
	  repaint();
      }
  }

public void dispose()
  {
    if (thread != null) thread.destroy();
    super.dispose();
  }

public boolean  mouseDown(Event ev, int x, int y)
  {
    userstopped = !userstopped;
    if (userstopped)
      stop();
    else
      {
	second_buffer_size.width = -1;
	start();
      }
    return true;
  }

public void hide()
  {
    second_buffer_size.width = -1;
    stop();
    super.hide();
  }

public void show()
  {
    if (!isVisible())
      start();
    super.show();
  }

public void setScale(int newlen)
  {
    if (length != newlen)
      {
	second_buffer_size.width = -1;
	length = newlen;
      }
  }

public void setWidth(double width)
  {
    this.width = width;
  }

public void setTextoffset(double textoffset)
  {
    this.textoffset = textoffset;
  }

public void setVertical(boolean vertical)
  {
    if (this.vertical != vertical)
      {
	second_buffer_size.width = -1;
	this.vertical = vertical;
      }
  }

public void updateFromSpriggans(Vector spriggans, int time)
  {
    if (second_buffer == null) return;  // watch those startup timings
    if (!running) return;
    if (time <= lastupdate) return;
    lastupdate = time;

    while (resizing) 
      try { Thread.currentThread().sleep(10); } 
    catch (InterruptedException e) {}
    updating = true;
    while (resizing)
      try { Thread.currentThread().sleep(10); } 
    catch (InterruptedException e) {}

    
    // clear the stripe that is offscreen
    if (vertical)
      {
	second_bufferGC.setColor(Color.black);	
	second_bufferGC.fillRect(0, (int)Math.round((time%(length+1))*delta),
				 real_size.width, (int)(delta+1));
      }
    else
      {
	second_bufferGC.setColor(Color.black);
	second_bufferGC.fillRect((int)Math.round((time%(length+1))*delta),
				 0, (int)(delta+1), real_size.height);
      }
    
    double wid2 = (1-width)/2;
    for (int i=0; i<spriggans.size(); i++)
      {
	Spriggan spriggan = (Spriggan) spriggans.elementAt(i);
	if (juggler.controls.st_shownumbers.getState() && 
	    spriggan instanceof Hand)
	  {
	    Hand h = (Hand)spriggan;
	    if ((int)h.path.starttime != time) continue;   
	    if (h.zero && !juggler.pattern.throwingNow(h.hand, time))
	      continue;
	    
	    second_bufferGC.setColor(Color.white);
	    int y, x=(int)Math.round((time%(length+1))*delta);
	    if (vertical)
	      {
		y = x + fm.getAscent()/2;
		x = (int)((wid2-textoffset+(width+2*textoffset)*h.hand)*real_size.width)+
		  (h.hand-1)*fm.stringWidth(h.label);
	      }
	    else
	      {
		x -= fm.stringWidth(h.label)/2;
		y = (int)((wid2-textoffset+(width+2*textoffset)*h.hand)*real_size.height)+
		  h.hand*fm.getAscent();
	      }
	    second_bufferGC.drawString(h.label, x, y);		
	    if (time%(length+1) == 0)   // ahh, clipping
	      if (vertical)
		second_bufferGC.drawString(h.label, x, real_size.height +
					 fm.getAscent()/2);
	      else
		second_bufferGC.drawString(h.label, real_size.width-
					 fm.stringWidth(h.label)/2, y);
	  }
	if (spriggan instanceof Ball) 
	  {
	    Ball b = (Ball)spriggan;
	    second_bufferGC.setColor(b.color);
	    int tm = b.thrw.time - b.thrw.height;
	    int x = (int)Math.round((time%(length+1))*delta);
	    if (b.thrw.height == 0)   // new ball
	      continue;
	    if (b.thrw.isCro())   // crossing
	      if (vertical)
		{
		  int y = x;
		  x = (int)((1-wid2-width*b.thrw.hand)*real_size.width);
		  double dx = width*real_size.width*(2*b.thrw.hand-1)/
		    b.thrw.height;
		  second_bufferGC.drawLine
		    (x+(int)(dx*(time-tm)), y,
		     x+(int)(dx*(time+1-tm)), y+(int)delta);
		}
	      else
		{
		  int y = (int)((1-wid2-width*b.thrw.hand)*real_size.height);
		  double dy = width*real_size.height*(2*b.thrw.hand-1)/
		    b.thrw.height;
		  second_bufferGC.drawLine
		    (x, y+(int)(dy*(time-tm)),
		     x+(int)delta, y+(int)(dy*(time+1-tm)));
		}
	    else if (b.thrw.height == 2)  // 2
	      if (vertical)
		{
		  int y = x;
		  x = (int)((wid2+width*b.thrw.hand)*real_size.width);
		  second_bufferGC.drawLine(x, y, x, y+(int)delta);
		}
	      else
		{
		  int y = (int)((wid2+width*b.thrw.hand)*real_size.height);
		  second_bufferGC.drawLine(x, y, x+(int)delta, y);
		}
	    else   // interesting even
	      {
		x -= (int)Math.round((time-tm)*delta);
		if (vertical)
		  {
		    double b1 = real_size.width*wid2*(b.thrw.height-2)/length;
		    double y0 = (time-tm)*2/(double)b.thrw.height - 1;
		    double t1 = -Math.atan2(y0,Math.sqrt(1-y0*y0))*180/Math.PI;
		    y0 += 2/(double)b.thrw.height;
		    double t2 = -Math.atan2(y0,Math.sqrt(1-y0*y0))*180/Math.PI;
		    t2 -= t1;
		    if (b.thrw.hand == 0) {t1 = 180-t1; t2 = -t2;}

		    int y = x;
		    x = (int)((wid2+width*b.thrw.hand)*real_size.width)
		      -(int)b1;
		    second_bufferGC.drawArc
		      (x, y, 2*(int)b1, (int)Math.round(b.thrw.height*delta), 
		       (int)t1, (int)t2);
		  }
		else
		  {
		    double b1 = real_size.height*wid2*(b.thrw.height-2)/length;
		    double x0 = (time-tm)*2/(double)b.thrw.height - 1;
		    double t1 = Math.atan2(Math.sqrt(1-x0*x0),x0)*180/Math.PI;
		    x0 += 2/(double)b.thrw.height;
		    double t2 = Math.atan2(Math.sqrt(1-x0*x0),x0)*180/Math.PI;
		    t2 -= t1;
		    if (b.thrw.hand == 1) {t1 = -t1; t2 = -t2;}

		    int y = (int)((wid2+width*b.thrw.hand)*real_size.height)
		      -(int)b1;
		    second_bufferGC.drawArc
		      (x, y, (int)Math.round(b.thrw.height*delta), (int)(2*b1),
		       (int)t1, (int)t2);
		  }
	      }
	  }
      }
    updating = false;
  }

public void update(Graphics g)  
  {
    paint(g);
  }

public void paint(Graphics g)
  {
    Rectangle rur = ur.bounds(), rll = ll.bounds();
    int x = rur.x, y = rur.y, w = rll.x+rll.width-x, h=rll.y+rll.height-y;
    if (w != second_buffer_size.width || h != second_buffer_size.height)
      {  // size changed! realloc second buffer and clear
	while (updating) 
	  try { Thread.currentThread().sleep(20); } 
	catch (InterruptedException e) {}
	resizing = true;
	while (updating)
	  try { Thread.currentThread().sleep(20); } 
	catch (InterruptedException e) {}

	real_size = new Dimension(w, h);
	if (vertical)
	  {
	    delta = real_size.height/(double)length;
	    real_size.height += (int)delta;
	  }
	else
	  {
	    delta = real_size.width/(double)length;
	    real_size.width += (int)delta;
	  }
	second_buffer = createImage(real_size.width, real_size.height);
	second_bufferGC = second_buffer.getGraphics();
	second_buffer_size = new Dimension(w, h);
	second_bufferGC.setColor(juggler.bgcolor);
	second_bufferGC.fillRect(0, 0, real_size.width, real_size.height);
	if (fontsize != Math.min(second_buffer_size.height/8,18))
	  {
	    fontsize = Math.min(second_buffer_size.height/8,18);
	    font = new Font("Helvetica", Font.PLAIN, fontsize);
	    fm = g.getFontMetrics();  // second buffer doesn't have FM
	  }
	second_bufferGC.setFont(font);
	resizing = false;
      }

    double now = juggler.timer.now();
    int pos = (int)(delta*(now - Math.floor(now/(length+1))*(length+1)));

    if (vertical)
      {
	g.drawImage(second_buffer, x, y+second_buffer_size.height-pos, this);
	g.drawImage(second_buffer, x, y-pos-(int)delta, this);
      }
    else
      {
	g.drawImage(second_buffer, x+second_buffer_size.width-pos, y, this);
	g.drawImage(second_buffer, x-pos-(int)delta, y, this);
      }
    lastdraw = now;
  }
  
}
