/*
 * @(#)SpectralEmbedder.java	
 *
 * Copyright (c) 1996,7 Allen Knutson, Matthew Levine, Gregory Warrington
 * This file is part of [MA][GNU]S.
 * 
 * [MA][GNU]S is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * [MA][GNU]S is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with [MA][GNU]S; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/**
 * @author  Allen Knutson, Matthew Levine, Gregory Warrington
 * @version $Id: SpectralEmbedder.java,v 1.6 1997/03/16 19:36:27 gwar Exp $
 */

import java.lang.*;
import java.util.Hashtable;
import java.util.Enumeration;

public class SpectralEmbedder implements Runnable
{
  Node nodes[];
  int n;
  Thread thread;
  int maxd;
  double epsilon = 1e-8;
  EVector vecs[];
  StateGraph stategraph;
  boolean running;


public SpectralEmbedder(Hashtable nodes, StateGraph stategraph)
  {
    Node tmp;

    this.stategraph = stategraph;
    Enumeration e=nodes.elements();
    n = nodes.size();
    if (n < 3) 
      {
	stategraph.showStatus("");
	return;
      }
    this.nodes = new Node[n];
    int i = 0;
    while(e.hasMoreElements())
      {
	tmp = (Node)e.nextElement();
	if (!tmp.visible)
	  { 
	    n--;
	    continue;
	  }
	this.nodes[i] = tmp;
	this.nodes[i].id = i;
	i++;
      }
    vecs = new EVector[3];
    vecs[0] = new EVector(n);
    vecs[0].cnst();
    vecs[1] = new EVector(n);
    vecs[1].rand();
    vecs[2] = new EVector(n);
    vecs[2].rand();

    thread = new Thread(this);
    thread.setPriority(Math.max(Thread.MIN_PRIORITY,thread.getPriority()-2));
    running = true;
    thread.start();
  }

public void run()
  {
    try 
      {
	EVector tmp = new EVector(n);

	for (int i=0; i<n; i++)
	  tmp.vals[i] = 0;
	for (int i=0; i<n; i++)
	  for(Enumeration en=nodes[i].edges.elements(); en.hasMoreElements();)
	    {
	      Edge ed = (Edge)en.nextElement();
	      if (ed.end.id == -1) continue;
	      tmp.vals[i]++;
	      tmp.vals[ed.end.id]++;
	    }

	maxd = 0;
	for (int i=0; i<n; i++)
	  if (tmp.vals[i] > maxd)
	    maxd = (int)tmp.vals[i];
	
	for(int i=1; i<=2; i++)
	  {
	    do
	      {
		thread.sleep(0);        // (let other things run)
		mult(vecs[i], tmp);     // apply laplacian
		EVector tmp2 = vecs[i]; vecs[i] = tmp; tmp = tmp2;
		for(int j=0; j<i; j++)  // make orthogonal to other evecs
		  vecs[i].orthogonalize(vecs[j]);
		vecs[i].renorm();       // renormalize
//		System.out.print(i+": "); vecs[i].print(); 
	      } while (vecs[i].normSqrOfDifference(tmp) > epsilon);
	  }
	apply();
      }
    catch (InterruptedException e) { }
    running = false;
  }

public void apply()
  {
    double xmin=Double.POSITIVE_INFINITY, xmax=Double.NEGATIVE_INFINITY, 
      ymin=Double.POSITIVE_INFINITY, ymax=Double.NEGATIVE_INFINITY;
    for(int i=0; i<n; i++)
      {
	if (vecs[1].vals[i] < xmin) xmin = vecs[1].vals[i];
	if (vecs[1].vals[i] > xmax) xmax = vecs[1].vals[i];
	if (vecs[2].vals[i] < ymin) ymin = vecs[2].vals[i];
	if (vecs[2].vals[i] > ymax) ymax = vecs[2].vals[i];
      }
    double xscale = 0.9*stategraph.d.width/(xmax-xmin);
    double yscale = 0.9*stategraph.d.height/(ymax-ymin);
    double xmid = (xmin+xmax)/2;
    double ymid = (ymin+ymax)/2;
    for(int i=0; i<n; i++)
      {
	nodes[i].coord.x = (int)((vecs[1].vals[i]-xmid)*xscale);
	nodes[i].coord.y = (int)((vecs[2].vals[i]-ymid)*yscale);
      }
    stategraph.showStatus("");
    stategraph.repaint();
  }

public void stop()
  {
    if (thread != null) thread.stop();
  }

private void mult(EVector e, EVector ans)
  {
    for (int i=0; i<n; i++)
      ans.vals[i] = 2*maxd*e.vals[i];  /* why the 2? */
    for (int i=0; i<n; i++)
      for(Enumeration en = nodes[i].edges.elements(); en.hasMoreElements(); )
	{
	  Edge ed = (Edge)en.nextElement();
	  if (ed.end.id == -1) continue;
	  ans.vals[i] += e.vals[ed.end.id];
	  ans.vals[ed.end.id] += e.vals[i];
	  ans.vals[i] -= e.vals[i];
	  ans.vals[ed.end.id] -= e.vals[ed.end.id];
	}
  }
}
