/*
 * @(#)State.java	
 *
 * Copyright (c) 1996,7 Allen Knutson, Matthew Levine, Gregory Warrington
 * This file is part of [MA][GNU]S.
 * 
 * [MA][GNU]S is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * [MA][GNU]S is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with [MA][GNU]S; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/**
 * @author  Allen Knutson, Matthew Levine, Gregory Warrington
 * @version $Id: State.java,v 1.31 1997/03/24 23:04:46 gwar Exp $
 */

// NOTES & TODO

// 1) The state diagram should really be updated every second as the
// digraph will want to know even if no ball has landed.  so a call to
// shiftstate should be put somewhere in the timing loop and the
// shiftstate call should be pulled out of nextthrw
// 2) CLARIFICATION:  i have to look and see how it is currently
// implemented, but the design is that _as soon as a time is reached_ the
// state should be shifted, so when a throw is found, the height-1 is
// what should be sent to addLanding.
// 3) A fixed amount of space is allocated for the landAt array.  We
// just treat the array as a loop.  The 'offset' variable keeps track of
// where in the array we are pretending is the beginning.  So whenever we
// access a time in the array we just mod by the arrLen.  Finding
// transitions requires that the arrLen be significantly higher than the
// highest throw we expect.
// 4) LandAt contains an array of landing data for each hand.  For an
// ASynch pattern, every other entry for a particular hand should be -1.
// The parity keeps track of which half of entries for a hand should
// be ignored.
// 5) State.copy() doesn't need to be that stupid does it?
// 6) firstAvail() and firstUnderfed() no longer modify their
// arguments, Pattern needs to be told of this.

import java.io.*;
import java.util.Vector;

// we just allocate an array corresponding to landing states at future
// times and then continually update which array element corresponds
// to current time
class State
{
  static private boolean DBG = false;
  static public char SEPCHAR = ':'; // for synchronous patterns
  static public char MTCHAR = '-';  // for no ball landing
  static public char ONECHAR = 'x'; // for one ball landing
  static public char SWITCHCHAR = '|'; // switching synchrony

  private boolean multiplex; // whether an element of landAt[][] can be > 1
  boolean syn[];     // synchrony of ith swi segment
  int swi[];         // amount of time in future for sync switch to occur.
  int par[];         // this keeps track of parity for each span of swi[i].
  // for a sync segment swi[i], par[i] says whether hand 0 throws at swi[i]
  // or one beat later.  for an async segment, (par[i] + swi[i])%2 denotes
  // the hand throwing at swi[i].  swi[i] is always 0.  this way lastswi
  // is never negative.  in addition, we always know the parity since we can
  // look at par[i].
  int lastswi;       // swi[lastswi] == maxLastLand +1

  private int numHands;    // length of landAt array  
  int arrLen;      // length of landAt[] array
  int lastLand[];  // the last landing time for a particular hand
                           // the following two variables are -1 if no
                           // ball is landing
  // int maxLastLand; // max over all hands of lastLand
  int landAt[][];          // landAt[hand][time] number of balls landing
                           // initialized to -1 as 0 means that no
                           // ball is landing there but the hand is
                           // marked as empty so we are not free to
                           // throw there (unless we're transitioning)
  int offset;   // which index in landAt array correpsonds to ceil(now).
  int maxHeight; // what is the maximum time in the future it can handle

  public State(int max)
    {
      multiplex   = true;  // allowed
      numHands    = 2;     // changing this will probably break much.
      // Note the lack of a method to change it manually.
      maxHeight   = max;   // the *2 forces the value of be even
      // and is for general safety
      arrLen      = 4*maxHeight;
			   
      lastswi = 0; // no balls landing
      syn         = new boolean[arrLen];
      swi         = new int[arrLen];
      par         = new int[arrLen];
      swi[0] = par[0] = 0;
      syn[0] = true;
      // don't need to initialize rest as we have lastswi and par gets set 
      // every time a ball lands.

      offset = arrLen;     // could set it to 0, but we don't want it
                           // to go negative as java is stupid.
      landAt = new int[numHands][arrLen];
      lastLand = new int[numHands];
      for(int i=0; i<numHands; i++)
	{
	  for(int j=0; j<arrLen; j++)
	    landAt[i][j] = -1;
	  lastLand[i] = -1;
	}
    }

public void printState()
  {
    for(int i = 0; i <= lastswi; i++)
      System.out.println(i+" swi: "+swi[i]+" par: "+par[i]+" syn: "+syn[i]);
    System.out.println(landingList());
  }

public void displayState()
{
// System.out.print("multiplex: "+multiplex+"  sync: "+sync);
// System.out.print("parity: "+parity+"  numHands: "+numHands);
// System.out.println("arrLen: "+arrLen+"   offset: "+offset);
// System.out.print("maxLastLand: "+maxLastLand+"   ");
// System.out.println(landingList() + "   " + label() + "swi: " + swi);
// for(int i=0; i<numHands; i++)
//   {
//    System.out.print("  " + i + ":  ");
//    for(int j=0; j<=maxLastLand; j++)
//      if(landAt[i][(j+offset)%arrLen] == -1)
//        System.out.print("-");
//      else
//        System.out.print("x");
//    System.out.println();
//   }
System.out.println(landingList());

}

// NOTE: ans.arrLen & ans.maxHeight get set in State(maxHeight)
// NOTE: offset should start off at 0
public State copy(int max)
    {
      State ans = new State(max);

//      System.out.println("in copy......................");
      //      if(DBG) this.displayState();

      ans.multiplex = this.multiplex;
      ans.lastswi = this.lastswi;
      for(int i=0; i <= this.lastswi; i++)
	{
	  ans.syn[i] = this.syn[i];
	  ans.swi[i] = this.swi[i];
	  if(ithSync(i))
	    ans.par[i] = this.par[i];
	  else
	    ans.par[i] = (this.par[i] + this.offset)%2;
	}
	  
      ans.numHands = this.numHands;
      if(this.swi[this.lastswi] > ans.arrLen)
      {
        System.out.println("ERROR: copy of state into one that is too small");
	System.out.print("this.maxLastLand: "+this.swi[this.lastswi]);
	System.out.println("ans.arrLen: "+ans.arrLen);
      }
      
      for(int i=0; i<this.numHands; i++)
	{
	  for(int j=0; j < this.swi[this.lastswi]; j++)
	    ans.landAt[i][j] = this.landAt[i][(j+this.offset)%this.arrLen];
	  ans.lastLand[i] = this.lastLand[i];
	}

      // if(DBG) ans.displayState();
      return ans;
    }

public void setMultiplex(boolean isMultiplex)
    {
      multiplex = isMultiplex;
    }

public boolean ithSync(int i)
  {
    return syn[i];
  }

public void setSync(boolean isSync)
{
  syn[0] = isSync;
}

  public boolean isSync()
    {
      return syn[0];
    }

  public boolean isMultiplex()
    {
      return multiplex;
    }

  public int getLandAt(int hand, int time)
    {
      return landAt[hand][time];
    }

  public int countBalls()
    {
      int num = 0;

      for(int i = 0; i < 2; i++)
	for(int j = 0; j <= lastLand[i]; j++)
	  num += Math.max(landAt[i][(offset + j) % arrLen], 0);
      return num;
    }

  private void resize(int newhgt)
    {
      State tmpSt = this.copy(this.maxHeight);

      this.maxHeight   = newhgt;   // the *2 forces the value of be even
                                   // and is for general safety
      this.arrLen      = 2*this.maxHeight;
			   
      this.lastswi = tmpSt.lastswi;
      this.syn = new boolean [this.arrLen];
      this.swi = new int[this.arrLen];
      this.par = new int[this.arrLen];
      for(int i=0; i<=this.lastswi; i++)
	{
	  this.syn[i] = tmpSt.syn[i];
	  this.swi[i] = tmpSt.swi[i];
	  if(ithSync(i))
	    this.par[i] = tmpSt.par[i];
	  else
	    this.par[i] = (tmpSt.par[i] + tmpSt.offset)%2;
	}

      this.offset = this.arrLen;     // could set it to 0, but we don't want it
                                     // to go negative as java is stupid.
      this.landAt = new int[tmpSt.numHands][this.arrLen];
      for(int i=0; i<tmpSt.numHands; i++)
	for(int j=0; j < tmpSt.swi[tmpSt.lastswi]; j++)
	  this.landAt[i][j] = tmpSt.landAt[i][(j+tmpSt.offset)%tmpSt.arrLen];
    }

  // the time of the hole is relative to starttime
  public static Vector findHoles(State curState, State newState, 
                                 int maxNeeded, int transTime)
  {
    int i = 0;
    boolean foundUnderfed = true;
    Vector ans = new Vector(maxNeeded);
    Place tempPlace;
    State tempNew = newState.copy(newState.maxHeight);
    State tempOld = curState.copy(curState.maxHeight);

    //curState.printState();
    //newState.printState();
    //System.out.println();

    tempOld.shiftState(transTime, false);

    while(i < maxNeeded && foundUnderfed)
      {
        tempPlace = tempOld.firstUnderfed(tempOld, tempNew);
        if(tempPlace.valid())
	  {
	    ans.addElement(tempPlace);
	    tempNew.removeLanding(tempPlace.hand(),  tempPlace.time(), true);
	  }
	else
	  foundUnderfed = false;
	i++;
      }

    //    tempOld.shiftState(-transTime, false);

    if(DBG)
      {
	System.out.println("list of holes we can throw to");
	for(i=0; i<ans.size(); i++)
	  {
	    System.out.print("...hnd: "+((Place)ans.elementAt(i)).hand());
	    System.out.println("  tm: "+((Place)ans.elementAt(i)).time());
	  }
      }

    return ans;
  }
 
  // gets list of balls we have available to throw from. the time of
  // the available ball is relative to transstarttime.
  public Vector findAvail(int maxNeeded, int maxTime)
  {
    int i = 0;
    boolean foundavail = true;
    Vector ans = new Vector(maxNeeded);
    Place tempPlace;
    State tempOld = copy(maxHeight);

    while(i < maxNeeded && foundavail)
      {
	if (DBG) System.out.println(tempOld.landingList());
        tempPlace = tempOld.firstAvail();
        if(tempPlace.valid() && tempPlace.time() < maxTime)
	  {
	    ans.addElement(tempPlace);
	    tempOld.removeLanding(tempPlace.hand(),  tempPlace.time(), true);
	  }
	else
	  foundavail = false;
	i++;
      }

    if(DBG)
      {
	System.out.println("list of places we can throw from");
	for(i=0; i<ans.size(); i++)
	  {
	    System.out.print("...hnd: "+((Place)ans.elementAt(i)).hand());
	    System.out.println("  tm: "+((Place)ans.elementAt(i)).time());
	  }
      }

    return ans;
  } // findAvail

  // change current time in state by amt seconds.  don't initialize
  // when backing up as this is usually reversing a temporary change.
  public boolean shiftState(int amt, boolean clobber)
    {
      // printState();

      //     System.out.println("............." + landingList() + 
      //			" " + lastswi + " " + swi[lastswi]);

     offset = (offset + amt + arrLen)%arrLen;
     for(int i=0; clobber && i < numHands; i++)
       lastLand[i] = Math.max(lastLand[i]-amt,-1);

     // we need to be careful as we could be skipping over more than one 
     // swi at a time.  ie, this code is buggy.
     if(true)
       {
	 int ll=0;    // first one that is not totally clobbered
	 while(ll < lastswi && swi[ll+1] - amt <= 0)
	   ll++;
	 for(int i = 0; i <= lastswi - ll; i++)
	   {
	     swi[i] = Math.max(swi[i+ll] - amt,0);
	     par[i] = par[i+ll];
	     syn[i] = syn[i+ll];
	   }
	 //	 if(DBG && ll == lastswi)
	 //	   System.out.println("sin::::::::::::" + syn[0]);
// 	 if(countBalls() > 0)
// 	   lastswi = Math.max(lastswi - ll,1);
// 	 else
	 lastswi = Math.max(lastswi - ll,0);
	 if(ll == 0 && syn[0])
	   par[0] = (par[0] + amt + 2)%2;
       }

     return true;  // valid shift
    } // shiftState


// only call this ugly method on async states.
public int eve()
{
  for(int i = 0; i < swi[Math.min(1,lastswi)]; i++)
    {
      if(landAt[0][i] > 0)
	return i%2;
      if(landAt[1][i] > 0)
	return (i+1)%2;
    }
  return 0;
}

  public boolean addLanding(int hand, int index, boolean nonzero)
    {
      // System.out.println("addLanding: "+hand+" "+index+" "+nonzero);
      if(index > maxHeight)
	resize(index);

      // initialize new landing spots
      for(int i=swi[lastswi]-1; i<index; i++)
	for(int j=0; j < numHands; j++)
	  landAt[j][(offset + i + 1)%arrLen] = -1;

      // can't have more than one ball in a hand at a time
      if(multiplex == false && landAt[hand][(offset + index)%arrLen] > 0)
	  {
	    System.out.println("ERROR: Invalid throw, not allowed to multiplex");
	    return false;
	  }

      if(landAt[hand][(offset + index)%arrLen] == 0)
	{
	  System.out.println("ERROR: Invalid throw, hand marked as empty");
	  return false;
	}

      // if we don't have anything in the state we want to initialize
      // the parity.  NOTE: assumes (rightfully so :) that arrLen is even.

      // actually add the landing
      if(nonzero == true)
	if(landAt[hand][(offset + index)%arrLen] == -1)
	  landAt[hand][(offset + index)%arrLen] = 1;
	else
	  landAt[hand][(offset + index)%arrLen]++;	  
      else
	landAt[hand][(offset + index)%arrLen] = 0;
      lastLand[hand] = Math.max(lastLand[hand], index);

      if(lastswi == 0)
	{ 
	  lastswi = 1;
	  syn[lastswi] = syn[0] = isSync();
	  swi[lastswi] = index + 1; 
	}
      else
	swi[lastswi] = Math.max(swi[lastswi], index+1);

      //System.out.println("lastswi: " + lastswi + "swi[]: " + swi[lastswi]);
//      if (DBG) System.out.print("after adding landing ml0:   " + lastLand[0]);
//      System.out.println("   ml1: "+lastLand[1]+"        "+landingList()); 


      setParity(index,hand);
      //System.out.println("addl: " + landingList() + " syn: " + syn[0]);

     return true;
    }

  // perhaps this method should only warn rather than refuse if
  // request is not consistent with data.
  public boolean removeLanding(int hand, int index, boolean nonzero)
    {
     int time = (index + offset)%arrLen;

     if(nonzero == false && landAt[hand][time] != 0)
       {
//	System.out.println("Trying to remove a 0 from a nonempty hand.");
        return false;
       }
     if(nonzero == true && landAt[hand][time] == 0)
       {
//        System.out.println("Trying to remove a ball from an empty hand.");
        return false;
       }
     if(landAt[hand][time] == 1)
       landAt[hand][time] = -1;
     else
       landAt[hand][time]--;

     return true;
    }

public Place nthEmpty(int acceptNum)
{			
  int delta = 1, tm = 0, numFound = 0;
  int hand, time;
  int j = -1, i;

  //  if(lastswi == 0)
  //    return new Place(0,0);

  if(ithSync(Math.max(lastswi-1,0)))
    {
      delta = 2;
      tm = swi[Math.max(lastswi-1,0)] + par[Math.max(lastswi-1,0)];
    }
  else
    tm = swi[Math.max(lastswi-1,0)];

  //if(DBG) System.out.println("----------starting to search " + landingList());
  //if(DBG) System.out.println("----------offset: " + offset + "  parity set to: " + par[lastswi]);
  tm += delta*acceptNum;
  numFound = acceptNum;
  while(true)
    {
      for(i = 0; i < delta; i++)
	{
	  if(ithSync(Math.max(lastswi-1,0)))
	    hand = (offset + tm + i) % numHands;
	  else
	    hand = (offset + tm + par[Math.max(lastswi-1,0)] + i) % numHands;
	  time = (offset + tm) % arrLen;
	  	  if((tm > lastLand[hand]) || 
	  	     (landAt[hand][time] < 0 && tm <= lastLand[hand]))
	    if(++numFound >= acceptNum)
	      {
		if(DBG) System.out.println(
			"Accepting hand: "+hand+"   time:" +tm);
		return(new Place(hand, tm));
	      }
	  //if(DBG) System.out.println("Rejecting hand: "+hand+"   time:" +tm);
	}
      tm += delta;
    }

}

  // finds first non-empty hand and returns an array consisting of
  // which hand this corresponds to and the time in the future
  public Place firstAvail()
    {
      Place tempPlace = new Place();

      for(int i=0; i < swi[lastswi]; i++)
 	 for(int j=0; j<numHands; j++)
	    if(landAt[j][(i+offset)%arrLen] > 0)
	      {
	       tempPlace.setHand(j);
	       tempPlace.setTime(i);
	       return tempPlace;
	      }

      return tempPlace;
    } // firstAvail

  // assumes a is a valid subset of b.  in asynch patterns this will 
  // check even the hands on the wrong parity (and off times for synch
  // patterns, but that's probably not that big of a deal - ie, you fix it.
  public Place firstUnderfed(State a, State b)
    {
      int aTime, bTime, tmpA;
      Place tempPlace = new Place();

      //System.out.println("In firstunderfed");
      //     a.printState();
      //b.printState();

      for(int i=0; i < b.swi[b.lastswi]; i++)
        for(int j=0; j<numHands; j++)
          {
	   aTime = (i + a.offset)%a.arrLen;
	   bTime = (i + b.offset)%b.arrLen;
           if(i < a.swi[a.lastswi]) 
             tmpA = a.landAt[j][aTime];
           else
             tmpA = -1;  // landAt[j][bTime] is definitely underfed!
	   if(b.landAt[j][bTime] > 0 && b.landAt[j][bTime] > tmpA)
	      {
	       tempPlace.setHand(j);
	       tempPlace.setTime(i);
		if(DBG)
	  System.out.println("underfed ball at hand: "+j+" time: "+i);
		return tempPlace;
	      }
	  }

      if(DBG) System.out.println("nothing underfed");
      return tempPlace;
    }  // firstUnderfed

  // seeing if a is a subset of b
  public static boolean isSubset(State a, State b)
    {
      // probably should give a more useful error message here
      if(a.numHands != b.numHands) 
	{
//	  System.out.print("You can add and subtract balls but");
//	  System.out.println(" you can't add or subtract a hand (yet)");
	  return false;
	}

       if(false) 
	 {
	   System.out.print("here are states being compared: ");
	   System.out.print(a.landingList()+"   ");
	   System.out.println(b.landingList());
	 }

      for(int i=0; i<a.numHands; i++)
	for(int j=0; j < a.swi[a.lastswi]; j++)
	  {
	    if(j >= b.swi[b.lastswi])
	      return false;
	    if(a.landAt[i][(j+a.offset)%a.arrLen] > 0 &&
	       b.landAt[i][(j+b.offset)%b.arrLen] <
	       a.landAt[i][(j+a.offset)%a.arrLen])
	      return false;
	  }
      if(DBG) System.out.println("found a subset");

      return true;
    }

  // claims to be written for general number of hands but probably
  // won't work in the intended manner for more than two hands.
  // NOTE: NOT FINISHED & really stupid.
  public String landingList()
    {
      StringBuffer str = new StringBuffer();
      int tmp;

      if(ithSync(0) == true)
	for(int i=0; i < swi[lastswi]; i++)
	  {
	    str.append('(');
	    for(int j=0; j < numHands; j++)
              {
		tmp = landAt[j][(offset+i)%arrLen];
		
		if(tmp > 0)
		  str.append((char)('0' + tmp));
		else if(tmp == 0)
		  str.append('-');
		else
		  str.append('_');
              }
	    str.append(')');
	  }
      else
	{
	// okay, so i got lazy....
	for(int i=0; i < swi[lastswi]; i++)
	  {
	    tmp = landAt[0][(offset+i)%arrLen];
	    if(tmp > 0)
	      str.append((char)('0' + tmp));
	    else if(tmp  == 0)
	      str.append('-');
	    else
	      str.append('_');
	  }
	str.append(':');
	for(int i=0; i < swi[lastswi]; i++)
	  {
	    tmp = landAt[1][(offset+i)%arrLen];
	    if(tmp > 0)
	      str.append((char)('0' + tmp));
	    else if(tmp  == 0)
	      str.append('-');
	    else
	      str.append('_');
	  }
      }
      return str.toString();
    } // landingList

//////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////

public void setParity(int time, int hand)
  {
    int i = 0;

    while(swi[i+1] <= time)      i++;

    if(ithSync(i))
      par[i] = (swi[i] + time) % 2;
    else
      par[i] = (offset + time + hand) % 2;

//      if(DBG) {
//        System.out.println("setParity(goo) called "+offset+" "+time+" "+hand);
//        displayState();
//        System.out.println("offset: " + offset + "  Parity: "+ par[i]);
//        //      System.out.println(label());
//      }
  }

//////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////

public static String getLabel(String str)
{
  String str1 = new String(str);
  String str2 = new String(str);
  StringBuffer str3;

  while(str1.length() > 0)
    {
      str2 = str1.substring(0, Math.max(str1.indexOf(SWITCHCHAR), 0));
      if(str2.length() > 0)
	if(str2.indexOf(SEPCHAR) > -1)
	  {
	    getSyncLabel(str2, 0);
    	    getSyncLabel(str2, 1);
	  }
      else
	getASyncLabel(str2);
    }
  //  if(DBG)
  //    System.out.println("getLabel returning: " + str);
  return str1; // fake
}

public static String getSyncLabel(String str, int hand)
  {
    int mid = str.indexOf(SEPCHAR);
    int swit = str.indexOf(SWITCHCHAR);
    String tmpstr = new String();

    //System.out.println(mid + "      " + str);
    if(mid < 0)
      return "";

    if(hand == 0)
      if(swit < mid)
	tmpstr = str.substring(swit + 1, mid);
      else
	tmpstr = str.substring(0, mid);
    else
      if(swit > mid)
	tmpstr = str.substring(mid+1, swit);
      else
	tmpstr = str.substring(mid+1);
    //    if(DBG)
    //System.out.println("getSyncLabel returning: " + tmpstr);
    return tmpstr;
  }

public static String getASyncLabel(String str)
  {
    int swit = str.indexOf(SWITCHCHAR);
    int mid = str.indexOf(SEPCHAR);
    String tmpstr = new String();

    if(swit < 0)
      tmpstr = str;
    else
      if(mid < swit)
	tmpstr = str.substring(swit + 1);
      else
	tmpstr = str.substring(0, swit);
    //    if(DBG)
    //      System.out.println("getASyncLabel returning: " + tmpstr);
    return tmpstr;
  }


// if state is both and sync then can't use maxLastLand to find out what
// the adjustment is for printing so we calculate it here.
private int findSyncAdjust(int st, int end)
  {
    for(int i = st; i < end; i++)
      if(landAt[0][(offset + i + arrLen) % arrLen] > -1 || 
	 landAt[1][(offset + i + arrLen) % arrLen] > -1)
	return i%2;
    return 0;
  }

private StringBuffer labelASync(int start, int end, int parity)
  {
    StringBuffer str = new StringBuffer();
    int hand, i, j, tmpparity = 0;;
    boolean found= false;

    //    if(DBG)
    //      System.out.println("labelASync: st: " + start + " end: " + end);

//     for(i = start; i < end && !found; i++)
//       for(j = 0; j < 2 && !found; j++)
// 	if(landAt[j][(i + offset) % arrLen] > -1)
// 	  {
// 	    tmpparity = (offset + i + j) % 2;	  
// 	    found=true;
// 	  }

      for(j=start; j < end; j++)
	{
	  hand = (offset + j + parity) % numHands;
	  if(j <= lastLand[hand])
	    if(landAt[hand][(j + offset)%arrLen] > 1)
	      str.append((char)('0' + landAt[hand][(offset+j)%arrLen]));
	    else if(landAt[hand][(j + offset)%arrLen] == 1)
	      str.append(ONECHAR);
	    else
	      str.append(MTCHAR);
	  else
	    str.append(MTCHAR);
	}

      //      if(DBG)
      //    System.out.println("labelASync returning: " + str);
    return str;
  }

private StringBuffer labelSync(int start, int end)
{
  StringBuffer str = new StringBuffer();
  
  //  if(DBG)
  //    System.out.println("labelSync: st: " + start + " end: " + end);

  for(int i=0; i < 2; i++)
    {
      for(int j=start; j < end; j+=2)
	{
	  if(j <= lastLand[i])
	    if(landAt[i][(j + offset)%arrLen] > 1)
	      str.append((char)('0' + landAt[i][(offset+j)%arrLen]));
	    else if(landAt[i][(j + offset)%arrLen] == 1)
	      str.append(ONECHAR);
	    else
	      str.append(MTCHAR);
	  else
	    str.append(MTCHAR);
	}
      if(i == 0 && str.length() > 0)
	str.append(SEPCHAR);
    }
  //  if(DBG)
  //      System.out.println("labelSync returning: " + str);
  return str;
}

public void printdbglabel()
{
  String str = label();
  System.out.println(str.substring(0,Math.max(str.indexOf(':'),0)));
  System.out.println(str.substring(str.indexOf(':')+1));
}

// we need this because we could be in sync, then going to async, and
// then switching back to async all in one ugly state.
public String label()
{
  int i = 1;
  StringBuffer str1 = new StringBuffer("");

  // printState();

  if(ithSync(0))
    str1.append(labelSync(swi[0] + par[0], swi[1]));
  else
    str1.append(labelASync(swi[0], swi[1], par[0]));

  //  if(DBG) System.out.println("lastswi: " + lastswi);
  while(i < lastswi)
    {
      if(str1.length() > 0)
	str1.append(SWITCHCHAR);
      if(ithSync(i))
	str1.append(labelSync(swi[i] + par[i], swi[i+1]));
      else
	str1.append(labelASync(swi[i], swi[i+1], par[i]));
      i++;
    }

  //yo
  //System.out.println("asdf: " + swi[0] + " " +str1+ "  " + landingList());

  //, str.concat(swilabel(swi[i]), false); // want async-style label
  //	str.concat(swilabel(swi[i]), true);  // want sync-style label
  if(str1.length() > 0 && str1.charAt(str1.length()-1) == SWITCHCHAR)
    str1.setLength(str1.length()-1);

  // if(DBG)
  //  System.out.println("label returning: " + str1.toString());
  if(str1.length() == 0)
    if(isSync())
      str1.append(MTCHAR).append(SEPCHAR).append(MTCHAR);
    else
      str1.append(MTCHAR);

  // think we're running into trouble because last switch time is
  // after last ball landing which makes the state graph print out too
  // much.  however, the last char could validly be a '-' if the end
  // state is synchronous.
  while(str1.length() > 0 && 
	str1.charAt(str1.length()-1) == MTCHAR &&
	(str1.toString()).indexOf(SEPCHAR) <=
	(str1.toString()).indexOf(SWITCHCHAR))
    str1.setLength(str1.length()-1);
  return str1.toString();
}

// if we are in the middle of a transition from sync <--> async we may
// need to display the state partially as sync and partially as async.
// public String swilabel(int sta, boolean sy)
//   {
//     StringBuffer str, tmpstr;

    

//     if(swi > 0)
//       if(sy)
// 	{
// 	  str = labelSync(findSyncAdjust(swi), swi);
// 	  tmpstr = labelASync(swi, maxLastLand + 1);
// //	  if(tmpstr.length() > 0)
// 	  if(1 > 0)
// 	    {
// 	      str.append(SWITCHCHAR);
// 	      str.append(tmpstr);
// 	    }
// 	}
//       else
// 	{
// 	  str = labelASync(0, swi);
// 	  if(maxLastLand % 2 != swi % 2)
// 	    tmpstr = labelSync(swi + 1, maxLastLand + 1);
// 	  else
// 	    tmpstr = labelSync(swi, maxLastLand + 1);
// //	  if(tmpstr.length() > 0)
// 	  if(1 > 0)
// 	    {
// 	      str.append(SWITCHCHAR);	
// 	      str.append(tmpstr);
// 	    }
// 	}
//     else
//       if(sync)
// 	str = labelSync((maxLastLand + 2)%2, maxLastLand + 1);
//       else
// 	str = labelASync(0, maxLastLand + 1);

//     //if(DBG) System.out.println(str);
//     return str.toString();
//   }

///////////////////////////////////////////////////////
// it's turtles all the way down (starting here)
///////////////////////////////////////////////////////

  // returns maximum number of seconds from now a ball currently in 
  // the air will land for a specific hand.  -1 if none.  does not
  // distinguish between no balls landing for a hand and being fed
  // an invalid hand.  NOT CURRENTLY USED
  public int lastNonEmpty(int hand)
    {
      int cur = -1;

      if(hand > numHands)
	return -1;

      for(int i=0; i<arrLen/2; i++)
	if(landAt[hand][(i+offset)%arrLen] > 0) 
	  cur = i;
      return cur;
    }

  // returns maximum number of seconds from now a ball currently in
  // the air will land for all hands.  -1 if none.  NOT CURRENTLY USED.
  public int lastNonEmpty()
    {
      int cur = -1;

      for(int i=0; i<numHands; i++)
	cur = Math.max(lastNonEmpty(i),cur);
      return cur;
    }

      // so underfed doesn't find these and think we should throw into them.
//      for(int i=0; i<a.numHands; i++)
//	for(int j=0; j <= a.maxLastLand; j++)
//	  {
//	    if(a.landAt[i][(j+a.offset)%a.arrLen] > 0)
//	       b.landAt[i][(j+b.offset)%b.arrLen] -=
//	       a.landAt[i][(j+a.offset)%a.arrLen];
//	  }


  // note doesn't check to see if 'multiplex' is the same.
// needs to be fancier
public boolean equal(State st2)
  {	
    if(this.numHands != st2.numHands ||
       this.syn[0] != st2.syn[0] ||
       this.swi[this.lastswi] != st2.swi[st2.lastswi])
      return false;

    for(int i=0; i < this.numHands; i++)
      {	
	if(this.lastLand[i] != st2.lastLand[i])
	  return false;
	for(int j=0; j < lastLand[i]; j++)
	  if(this.landAt[i][(j+this.offset)%this.arrLen] != 
	     st2.landAt[i][(j+st2.offset)%arrLen])
	    return false;
      }		
    return true;
  }

public boolean canThrow(int th)
  {
    if(ithSync(0))
      if(th == 0)
	return(landAt[0][0] == -1 || landAt[1][0] == -1);
      else
	return(landAt[0][0] > 0 || landAt[1][0] > 0);	
    else
      if(th == 0)
	return(landAt[(offset + par[0]) % numHands][0] == -1);
      else
	return(landAt[(offset + par[0]) % numHands][0] > 0);
  }
    


// these don't do anything yet.
public State peekState(int hand, int hgt, boolean crossing)
  {
    State peekState = this.copy(this.maxHeight);
    int newHand;

    if(crossing)   newHand = (hand + 1)%2;
    else newHand = hand;

    if(peekState.addLanding(newHand, hgt, hgt != 0))
      return peekState;	
    else
      return null;
  }	
	

// this should only be called for asynchronous patterns
public State peekState(int hgt)
  {
    int hand = (this.offset + this.par[0] + hgt)%2;

    State peekState = this.copy(this.maxHeight);

    if(peekState.addLanding(hand, hgt, true))
      return peekState;	
    else
      return null;
  }	

// // to find out where switch occurs when swi > 0
// private int findSwitch()
//   {
//     int i;

//     if(swi <= 0)
//       return -1;

//     // sync one is pretty stupid (and wrong)
//     if(sync)
//       for(i = 0; i <= maxLastLand; i++)
// 	if(landAt[0][(offset + i + arrLen) % arrLen] < 0 && 
// 	   landAt[1][(offset + i + arrLen) % arrLen] < 0)
// 	  {
// 	    if(DBG) System.out.println("sync, switch is: " + i);
// 	    if(i == 0) both = false;
// 	    if(DBG) System.out.println("SETTING BOTH TO FALSE");
// 	    setParity();
// 	    return i;
// 	  }
//     else
//       for(i = 0; i <= maxLastLand; i++)
// 	{
// 	  int hand = (offset + i + parity + 1) % numHands; // off hand
// 	  if(landAt[hand][(offset + i + arrLen) % arrLen] > -1)
// 	    {
// 	      if(DBG) System.out.println("async, switch is: " + i);
// 	      if(i == 0) both = false;
// 	      if(DBG) System.out.println("SETTING BOTH TO FALSE");
// 	      setParity();
// 	      return i;
// 	    }
// 	}

//     return (maxLastLand + 1); // shouldn't be both anymore
//   }
}
