/*
 * @(#)StateGraph.java	
 *
 * Copyright (c) 1996,7 Allen Knutson, Matthew Levine, Gregory Warrington
 * This file is part of [MA][GNU]S.
 * 
 * [MA][GNU]S is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * [MA][GNU]S is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with [MA][GNU]S; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

// CONTROLS:
// left and right pattern entry fields
// vanilla entry
// drive juggler (randomly)
// grab states
// clear all
// clear irrelevant
// expand all compound edges
// change font size
// display '1' for 'x'
// allow multiplexes while in drive mode


/**
 * @author  Allen Knutson, Matthew Levine, Gregory Warrington
 * @version $Id: StateGraph.java,v 1.36 1997/03/23 14:49:46 gwar Exp $
 */

import java.applet.*;
import java.io.*;
import java.awt.*;
import java.util.Hashtable;
import java.util.Enumeration;
import java.util.Stack;
import java.util.Vector;

public class StateGraph extends Frame implements Runnable
{	
  static private boolean DBG = false;  

  final Color JUGGLERCOL = Color.blue;
  final Color CURRENTCOL = Color.red;
  final Color OTHERCOL = Color.white;
  final Color BOTHCOL = Color.magenta;

  final Color CREATECOL = Color.cyan;
  final Color DELETECOL = Color.green;
  final Color ZEROCOL = Color.gray;
  final Color NONZEROCOL = Color.black;

  Hashtable nodes;
  Hashtable edges;

  Node curNode;
  Node jugNode;
  Node center;
  Node selected;
  Node dummy;

  boolean running, needpaint;
  Thread thread;

  Image second_buffer;
  Graphics second_bufferGC;
  Dimension second_buffer_size;

  boolean driven;  // driven by juggler
  boolean groupie; // whether the curNode should follow the jugNode

  Button clearbutton;
  //TextField rightpat, leftpat;
  Panel controls;
  Label status;
  //  MessageWindow message;

  Dimension d = new Dimension();
  Point ori = new Point(0,0);

  Marker ur;

  char firstKey;
  char secondKey;

  SpectralEmbedder specem;

public StateGraph()
  {
    setTitle("State Graph");

    nodes = new Hashtable(20);
    edges = new Hashtable(20, 10);

    BallSequence seq = new BallSequence();
    dummy = new Node(seq.initState());
    dummy.visible = false;
    nodes.put(dummy.toKey(), dummy);

    curNode = dummy;
    jugNode = dummy;
    firstKey = secondKey = (char)-1;

    running = false;
    needpaint = false;
    driven = true;
    groupie = true;

    second_buffer_size= new Dimension();

    d = size();
    ori = new Point(0,0);
    
    resize(330, 230);

    setLayout(new BorderLayout());
    controls = new Panel();
    controls.setLayout(new BorderLayout());
    Panel dummy = new Panel();
    dummy.setLayout(new BorderLayout());
    dummy.add("East", clearbutton = new Button("Clear"));
    //    dummy.add("West", layoutbutton = new Button("Re-layout"));
    controls.add("East", dummy);
    controls.add("Center", status = new Label(""));
//    controls.add("West", leftpat = new TextField(6));
//    controls.add("Center", rightpat = new TextField(6));
    add("South", controls);
    add("North", ur = new Marker());
  }

public void newJugglerState(State st, Vector ths)
  {
    Node nd = addState(st);

    // WARNING
    //    if(jugNode == nd)
    //      return;

    nd.visible = true;
    if(jugNode != dummy || dummy.visible)
      addEdge(jugNode, nd, ths);
    setJugNode(nd);
    if(groupie)
      setCurNode(jugNode);

    needpaint = true;

//    System.out.println("----------------current sg------------------");
//    printStateGraph();
  }

  // delete all nodes that don't have current number of balls and all
  //corresponding edges.
public void delToLevel()
  {
    int curnum = jugNode.st.countBalls();
    Node nd;

    for(Enumeration n = nodes.elements(); n.hasMoreElements(); )
      {
	nd = (Node)n.nextElement();	
	if(nd.st.countBalls() != curnum && nd.visible)
	  removeNode(nd);
      }
  }

public Node addState(State st)
  {
    Node nd;

    //    System.out.println("sg:  ." + st.label() + ".");
    //System.out.println((Node)nodes.get(st.label()));
    if((nd = (Node)nodes.get(st.label())) == null)
      {
	if(DBG) 
	  System.out.println("can't find st: "+st.label()+"  "+st.landingList());
	State newst = st.copy(st.maxHeight);
//	if(DBG) System.out.println("can't find state: "+newst.label());
	nd = new Node(newst);

	nd.coord.x = (int)((Math.random() - 0.5)*d.width);
	nd.coord.y = (int)((Math.random() - 0.5)*d.height);

	nodes.put(nd.toKey(), nd);
      }
    return nd;
  }

public void setCurNode(Node nd)
  {
    if(curNode != nd)
      {
	if(curNode == jugNode)
	  curNode.setColor(JUGGLERCOL);
	else
	  curNode.setColor(OTHERCOL);
	curNode = nd;
	if(curNode == jugNode)
	  {
	    curNode.setColor(BOTHCOL);
	    groupie = true;
	  }
	else
	  {
	    curNode.setColor(CURRENTCOL);
	    groupie = false;
	  }
	center = curNode;
      }
    else
      {
	if(curNode == jugNode)
	  if(groupie)
	    groupie = false;
	  else
	    groupie = true;
	else
	  {
	    curNode.setColor(OTHERCOL);
	    curNode = dummy;
	  }
      }
  }

public void setJugNode(Node nd)
  {
    if(nd == null)
      {
	System.out.println("ERROR: null is not a valid node for juggler");
	return;
      }

    if(jugNode == curNode)
      jugNode.setColor(CURRENTCOL);
    else
      jugNode.setColor(OTHERCOL);

    jugNode = nd;
    if(jugNode == curNode)
      jugNode.setColor(BOTHCOL);
    else
      jugNode.setColor(JUGGLERCOL);      
  }

public boolean action(Event ev, Object arg)
  {
//     if (ev.target == leftpat || ev.target == rightpat)
//       {
// 	String str = ((TextField)ev.target).getText();
// 	Thrw th = PatternParser.getThrw(str);

// 	if(th != null)
// 	  {
// 	    if(ev.target == leftpat)
// 	      handleThrow(th, 0); // hand 0
// 	    else
// 	      handleThrow(th, 1); // hand 1
// 	    if (message != null) message.hide();
// 	  }
// 	else
// 	  if(!handleCommand(str.toUpperCase().charAt(0)))
// 	    {
// 	      if (message == null)
// 		message = new MessageWindow();
// 	      message.setMessage("Invalid throw/command");
// 	    }
// 	  else
// 	    if (message != null) message.hide();

// 	return true;   // handled event, don't pass it up further
//       }
    if (ev.target == clearbutton)
      {
	clearGraph();
	return true;
      }
    return false;
  }

private void clearGraph()
  {
    if (specem != null && specem.running) return;
    nodes.clear();
    edges.clear();
    nodes.put(dummy.toKey(), dummy);    
    jugNode = dummy;
    curNode = dummy;
    dummy.visible = false;
    needpaint = true;
  }

private boolean handleCommand(char ch)
  {
    switch(ch)
      {
      case 'c':
	if (DBG) System.out.println("handled a 'c'");
	clearGraph();
	break;
      case 'k': 
	if (DBG) System.out.println("handled a 'k'");
	removeNode(curNode);
	break;
      case 'd':
	if (DBG) System.out.println("handled a 'd'");
	handleThrow(new Thrw(-1, 0, 0));
	break;
      case 's':
	// spectral
	startRelayout();
	break;
      case 'l':
	delToLevel();
	break;
      case ' ':
	setCurNode(jugNode);
	groupie = true;
	break;
      default:
	return false; // didn't handle
	// implicit break;
      }
    return true;
  }

private void handleThrow(Thrw th)
  {
    Vector vc = new Vector(5,1);

    if(curNode.st.canThrow(th.height))
      {
	State newst = curNode.st.peekState(th.height);
	newst.shiftState(1, true);
	if(DBG) System.out.println("throw: " + th + 
				   " newst: " + newst.label());

	Node nd = addState(newst);
	vc.addElement(th);
	Edge ed = addEdge(curNode, nd, vc);
	setEdgeColor(ed, (char)th.height);
	if(nd != curNode)
	  setCurNode(nd);

	needpaint = true;
      }
  }

  public void hide()
  {
    stop();
    super.hide();
  }

  public void show()
  {
    if (!isVisible())
      start();
    super.show();
  }


  public void update(Graphics g)
  {
    paint(g);
  }
  
  public void paint(Graphics g)
  {
    Dimension d = size();
    if (d.width != second_buffer_size.width ||
        d.height != second_buffer_size.height)
      {
        second_buffer = createImage(d.width, d.height);
        second_bufferGC = second_buffer.getGraphics();
        second_buffer_size = d;
	//  g.clearRect(0, 0, d.width, d.height);
      }           

    //second_bufferGC.clearRect(0, 0, d.width, d.height); 
    mypaint(second_bufferGC, g.getFontMetrics());
    g.drawImage(second_buffer, 0, 0, this);
  }

public void mypaint(Graphics g, FontMetrics fm)
  {
    Node tmpNode, nd;
    Rectangle rur = ur.bounds(), rll = controls.bounds();
    int x = rur.x, y = rur.y;
    d.width = rll.x+rll.width-x;
    d.height = rll.y-y;
    ori.x = x+d.width/2;
    ori.y = y+d.height/2;

    g.setColor(Color.lightGray);
    g.fillRect(x, y, d.width, d.height);

    // move all nodes on screen
    for(Enumeration n = nodes.elements(); n.hasMoreElements(); )
      {
	nd = (Node)n.nextElement();
	if (nd.coord.x < -d.width/2) nd.coord.x = -d.width/2;
	if (nd.coord.x > d.width/2) nd.coord.x = d.width/2;
	if (nd.coord.y < -d.height/2) nd.coord.y = -d.height/2;
	if (nd.coord.y > d.height/2) nd.coord.y = d.height/2;
      }
    
    for(Enumeration n = nodes.elements(); n.hasMoreElements(); )
      {
	nd = (Node)n.nextElement();
//	if(nd.visible)
//	  for(Enumeration e = nd.edges.elements(); e.hasMoreElements(); )
//	    ((Edge)e.nextElement()).paint(g, d, fm);
	nd.paint(g, d, ori, fm);
      }	
//    for(Enumeration e = edges.elements(); e.hasMoreElements(); )
//      ((Edge)e.nextElement()).paint(g, d, fm);

  }

public synchronized boolean mouseDown(Event evt, int x, int y) 
  {
    double bestdist = Double.MAX_VALUE;
    boolean inNode = false;

    for (Enumeration e = nodes.elements(); e.hasMoreElements(); )
      {
	Node n = (Node)e.nextElement();
	double dist = (n.coord.x + ori.x - x) * (n.coord.x + ori.x - x) + 
	  (n.coord.y + ori.y - y) * (n.coord.y + ori.y - y);
	if(Math.abs(n.coord.x + ori.x - x) <= n.box.x &&
	   Math.abs(n.coord.y + ori.y - y) <= n.box.y &&
	   dist < bestdist)
	  {
	    selected = n;
	    bestdist = dist;
	    inNode = true;
	  }
      }

    if(!inNode)
      {
	setCurNode(dummy);
	selected = null;
	needpaint = true;
	return false;
      }

    selected.coord.x = x - ori.x;
    selected.coord.y = y - ori.y;
    if(curNode != selected)
      setCurNode(selected);
    needpaint = true;
    return true;
  }
  
public synchronized boolean mouseDrag(Event evt, int x, int y) 
  {
    if(selected == null)
      return false;

    selected.coord.x = x - ori.x;
    selected.coord.y = y - ori.y;
    needpaint = true;
    return true;
  }
  
public synchronized boolean mouseUp(Event evt, int x, int y) 
  {
    if(selected == null)
      return false;

    selected.coord.x = x - ori.x;
    selected.coord.y = y - ori.y;
    //	selected.fixed = selectedfixed;
    selected = null;
    
    needpaint = true;
    return true;
  }
  
public boolean keyDown(Event evt, int key)
  {
    
//     if((char)key == 'P')
//       {
// 	System.out.println("digraph test {");
// 	System.out.println("center=1|");
// 	System.out.println("size=*8.5,11*");
// 	  for(Enumeration e = nodes.elements(); e.hasMoreElements(); )
// 	    {
// 	      System.out.println("*" + ((Node)e.nextElement()).label + "* [shape=box]|");
// 	    }
// 	  for(Enumeration e = edges.elements(); e.hasMoreElements(); )
// 	    {
// 	      Edge ed = (Edge)e.nextElement();
// 	      System.out.println("*"+ed.start.label+ "*->*" + ed.end.label + 
// 				 "* " + "[label = *" + ed.th + "*" + "]");
// 	    }
// 	  System.out.println("}");
//       }

    if(handleCommand((char)key))
      return true;
    
    int th = isThrow((char)key);

    if(th >= 0)
      handleThrow(new Thrw(th, 0, 0));
    return true;
  }

public Vector transitionViaSG(Node nd1, Node nd2)
  {
    Stack path = new Stack();
    unvisitNodes();

    shortestPath(nd1, nd2, path);
    return (Vector) path;
  }


// okay, matt.  i know.  you don't do a depth first search when
// looking for a shortest path.
public boolean shortestPath(Node nd1, Node nd2, Stack seen)
  {
    Edge ed;
    int i = 0;

    if(nd1 == nd2)
      return true;

    if(nd1.visited)
      return false;

    nd1.visited = true;
    while(i < nd1.edges.size())
      {
	ed = (Edge) nd1.edges.elementAt(i);
	seen.push(ed.thrw);
	if(shortestPath(ed.end, nd2, seen))
	  return true;
	else
	  seen.pop();
	i++;
      }
    return false;
  }

public void unvisitNodes()
  {
    for(Enumeration e = nodes.elements(); e.hasMoreElements(); )
	((Node)e.nextElement()).visited = false;
  }

public void setEdgeColor(Edge ed, char ch)
  {
    if(isThrow(ch) > -1)
      if(ch == '0')
	ed.color = ZEROCOL;
      else
	ed.color = NONZEROCOL;	
    else
      switch(ch)
	{
	case 'C':
	  ed.color = CREATECOL;
	  break;
	case 'D':
	  ed.color = DELETECOL;
	  break;
	default:
	  ed.color = NONZEROCOL;
	  break;
	}
  }

private int isThrow(char ch)
  {
    // regular throw
    if((ch - '0' < 10) && (ch - '0' >= 0))
      return(ch - '0');
    // insanely high throw - pretend we're in base-35.  Note, 
    // you can only use letters up to 'w' as 'x' is for something
    // else.
    if((ch - 'A' < 26) && (ch - 'A' >= 0))
      return(ch - 'A' + 10);
    return -1;
  }  // PatternParser.isThrow

public void printStateGraph()
  {
    Node tmpNode;

    for(Enumeration e = nodes.elements(); e.hasMoreElements(); )
      {
	tmpNode = (Node)e.nextElement();
	tmpNode.printNode();
      }
  }

public Edge addEdge(Node nd1, Node nd2, Vector ths)
  {
    Edge ed = new Edge(nd1, nd2, ths);
    
    if((Edge)edges.get(ed.toKey()) == null)
      {
	edges.put(ed.toKey(), ed);
	nd1.addEdge(ed);
      }
    return ed;
  }

public void removeNode(Node nd)
  {
    Node tempNode;
    String edgeKey;

    if(jugNode == nd)
      jugNode = dummy;
    curNode = dummy;

    for(Enumeration e = nd.edges.elements(); e.hasMoreElements(); )
      edges.remove(((Edge)e.nextElement()).toKey());

    // need to go through and remove all the relevant edges.
    // would probably be better if nodes kept track of incoming edges.
    for(Enumeration n = nodes.elements(); n.hasMoreElements(); )
      {
	tempNode = (Node)n.nextElement();
	edgeKey = Edge.name(tempNode, nd);
	if(edges.containsKey(edgeKey))
	   {
	     tempNode.edges.removeElement(edges.get(edgeKey));
	     edges.remove(edgeKey);
	   }
      }

    nodes.remove(nd.toKey());

    needpaint = true;
  }

  void startRelayout()
  {
    if (specem != null)	specem.stop();
    showStatus("recomputing layout...");
    specem = new SpectralEmbedder(nodes, this);
  }


public void showStatus(String s)
  {
    status.setText(s);
  }

public void start()
  {
    if (running) return;
    running=true;
    if (thread == null) thread = new Thread(this);
    thread.start();
    thread.setPriority(thread.getPriority()-2);
  }

public void stop()
  {
    if (!running) return;
    running = false;
    thread.stop();
    thread = null;
  }

public void run()
  {
    while (running)
      {
	try { Thread.sleep(20); } catch (InterruptedException e) {return; }
	if (needpaint) repaint();
	needpaint = false;
      }
  }


}


